/*
 * Decompiled with CFR 0.152.
 */
package org.dcache.nfs.v4;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.security.Principal;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import javax.annotation.concurrent.GuardedBy;
import org.dcache.nfs.ChimeraNFSException;
import org.dcache.nfs.status.BadSessionException;
import org.dcache.nfs.status.BadStateidException;
import org.dcache.nfs.status.StaleClientidException;
import org.dcache.nfs.util.Cache;
import org.dcache.nfs.util.NopCacheEventListener;
import org.dcache.nfs.v4.ClientCache;
import org.dcache.nfs.v4.ClientRecoveryStore;
import org.dcache.nfs.v4.DefaultClientCache;
import org.dcache.nfs.v4.EphemeralClientRecoveryStore;
import org.dcache.nfs.v4.FileTracker;
import org.dcache.nfs.v4.NFS4Client;
import org.dcache.nfs.v4.NFS4State;
import org.dcache.nfs.v4.Stateids;
import org.dcache.nfs.v4.xdr.clientid4;
import org.dcache.nfs.v4.xdr.sessionid4;
import org.dcache.nfs.v4.xdr.stateid4;
import org.dcache.nfs.v4.xdr.verifier4;
import org.dcache.oncrpc4j.util.Bytes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NFSv4StateHandler {
    private static final Logger _log = LoggerFactory.getLogger(NFSv4StateHandler.class);
    private static final int STATE_INITIAL_SEQUENCE = 0;
    private final AtomicInteger _clientId = new AtomicInteger(0);
    private final ClientCache _clientsByServerId;
    private final ReentrantReadWriteLock _accessLock = new ReentrantReadWriteLock();
    private final Lock _readLock = this._accessLock.readLock();
    private final Lock _writeLock = this._accessLock.writeLock();
    private final int _leaseTime;
    private boolean _running;
    private final int _instanceId;
    private final FileTracker _openFileTracker = new FileTracker();
    private final ClientRecoveryStore clientStore;
    private final ScheduledExecutorService _cleanerScheduler;

    public NFSv4StateHandler() {
        this(90, 0, new EphemeralClientRecoveryStore());
    }

    public NFSv4StateHandler(int leaseTime, int instanceId, ClientRecoveryStore clientStore) {
        this(leaseTime, instanceId, clientStore, new DefaultClientCache(leaseTime, new DeadClientCollector(clientStore)));
    }

    public NFSv4StateHandler(int leaseTime, int instanceId, ClientRecoveryStore clientStore, ClientCache clientsByServerId) {
        this._leaseTime = leaseTime;
        this._clientsByServerId = clientsByServerId;
        this._running = true;
        this._instanceId = instanceId;
        this.clientStore = clientStore;
        this._cleanerScheduler = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("NFSv41 client periodic cleanup").setDaemon(true).build());
        this._cleanerScheduler.scheduleAtFixedRate(() -> this._clientsByServerId.cleanUp(), this._leaseTime * 4, this._leaseTime * 4, TimeUnit.SECONDS);
        this._cleanerScheduler.schedule(() -> clientStore.reclaimComplete(), (long)this._leaseTime, TimeUnit.SECONDS);
    }

    public void removeClient(NFS4Client client) {
        this._writeLock.lock();
        try {
            Preconditions.checkState((boolean)this._running, (Object)"NFS state handler not running");
            this._clientsByServerId.remove(client.getId());
            this.clientStore.removeClient(client.getOwnerId());
        }
        finally {
            this._writeLock.unlock();
        }
        client.tryDispose();
    }

    private void addClient(NFS4Client newClient) {
        this._writeLock.lock();
        try {
            Preconditions.checkState((boolean)this._running, (Object)"NFS state handler not running");
            this._clientsByServerId.put(newClient.getId(), newClient);
            this.clientStore.addClient(newClient.getOwnerId());
        }
        finally {
            this._writeLock.unlock();
        }
    }

    public NFS4Client getConfirmedClient(clientid4 clientid) throws StaleClientidException {
        NFS4Client client = this.getValidClient(clientid);
        if (!client.isConfirmed()) {
            throw new StaleClientidException("client not confirmed.");
        }
        return client;
    }

    public NFS4Client getValidClient(clientid4 clientid) throws StaleClientidException {
        NFS4Client client = this.getClient(clientid);
        if (!client.isLeaseValid()) {
            throw new StaleClientidException("client expired.");
        }
        return client;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NFS4Client getClient(clientid4 clientid) throws StaleClientidException {
        this._readLock.lock();
        try {
            Preconditions.checkState((boolean)this._running, (Object)"NFS state handler not running");
            NFS4Client client = this._clientsByServerId.get(clientid);
            if (client == null) {
                throw new StaleClientidException("bad client id.");
            }
            NFS4Client nFS4Client = client;
            return nFS4Client;
        }
        finally {
            this._readLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NFS4Client getClientIdByStateId(stateid4 stateId) throws ChimeraNFSException {
        this._readLock.lock();
        try {
            Preconditions.checkState((boolean)this._running, (Object)"NFS state handler not running");
            clientid4 clientId = new clientid4(Bytes.getLong((byte[])stateId.other, (int)0));
            NFS4Client client = this._clientsByServerId.get(clientId);
            if (client == null) {
                Optional<NFS4Client> first = this._clientsByServerId.peek().findFirst();
                if (first.isPresent()) {
                    NFS4Client s = first.get();
                    _log.warn("Unknown client " + clientId + " returning " + s.getId());
                    NFS4Client nFS4Client = s;
                    return nFS4Client;
                }
                throw new BadStateidException("no client for stateid: " + stateId);
            }
            NFS4Client nFS4Client = client;
            return nFS4Client;
        }
        finally {
            this._readLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NFS4Client getClient(sessionid4 id) throws ChimeraNFSException {
        this._readLock.lock();
        try {
            Preconditions.checkState((boolean)this._running, (Object)"NFS state handler not running");
            clientid4 clientId = new clientid4(Bytes.getLong((byte[])id.value, (int)0));
            NFS4Client client = this._clientsByServerId.get(clientId);
            if (client == null) {
                throw new BadSessionException("session not found: " + id);
            }
            NFS4Client nFS4Client = client;
            return nFS4Client;
        }
        finally {
            this._readLock.unlock();
        }
    }

    public NFS4Client clientByOwner(byte[] ownerid) {
        this._readLock.lock();
        try {
            NFS4Client nFS4Client = this._clientsByServerId.stream().filter(c -> Arrays.equals(c.getOwnerId(), ownerid)).findAny().orElse(null);
            return nFS4Client;
        }
        finally {
            this._readLock.unlock();
        }
    }

    public void updateClientLeaseTime(stateid4 stateid) throws ChimeraNFSException {
        Preconditions.checkState((boolean)this._running, (Object)"NFS state handler not running");
        NFS4Client client = this.getClientIdByStateId(stateid);
        NFS4State state = client.state(stateid);
        if (!state.isConfimed()) {
            throw new BadStateidException("State is not confirmed");
        }
        Stateids.checkStateId(state.stateid(), stateid);
        client.updateLeaseTime();
    }

    public List<NFS4Client> getClients() {
        this._readLock.lock();
        try {
            Preconditions.checkState((boolean)this._running, (Object)"NFS state handler not running");
            List<NFS4Client> list = this._clientsByServerId.peek().collect(Collectors.toList());
            return list;
        }
        finally {
            this._readLock.unlock();
        }
    }

    public NFS4Client createClient(InetSocketAddress clientAddress, InetSocketAddress localAddress, int minorVersion, byte[] ownerID, verifier4 verifier, Principal principal, boolean callbackNeeded) {
        NFS4Client client = new NFS4Client(this, this.nextClientId(), minorVersion, clientAddress, localAddress, ownerID, verifier, principal, TimeUnit.SECONDS.toMillis(this._leaseTime), callbackNeeded);
        this.addClient(client);
        return client;
    }

    public FileTracker getFileTracker() {
        return this._openFileTracker;
    }

    public boolean isGracePeriod() {
        Preconditions.checkState((boolean)this._running, (Object)"NFS state handler not running");
        return this.clientStore.waitingForReclaim();
    }

    public synchronized void reclaimComplete(byte[] owner) {
        this.clientStore.reclaimClient(owner);
    }

    public synchronized void wantReclaim(byte[] owner) throws ChimeraNFSException {
        this.clientStore.wantReclaim(owner);
    }

    @GuardedBy(value="_writeLock")
    private void drainClients() {
        this._clientsByServerId.stream().forEach(c -> {
            c.tryDispose();
            this._clientsByServerId.remove(c.getId());
        });
    }

    public void shutdown() throws IOException {
        this._writeLock.lock();
        try {
            Preconditions.checkState((boolean)this._running, (Object)"NFS state handler not running");
            this._running = false;
            this.drainClients();
            this._cleanerScheduler.shutdown();
            this.clientStore.close();
        }
        finally {
            this._writeLock.unlock();
        }
    }

    public synchronized boolean isRunning() {
        return this._running;
    }

    public int getInstanceId() {
        return this._instanceId;
    }

    public static int getInstanceId(stateid4 stateid) {
        long clientid = Bytes.getLong((byte[])stateid.other, (int)0);
        return (int)(clientid >> 16) & 0xFFFF;
    }

    private clientid4 nextClientId() {
        long now = System.currentTimeMillis() / 1000L;
        return new clientid4(now << 32 | (long)(this._instanceId << 16) | (long)(this._clientId.incrementAndGet() & 0xFFFF));
    }

    public stateid4 createStateId(NFS4Client client, int count) {
        byte[] other = new byte[12];
        Bytes.putLong((byte[])other, (int)0, (long)client.getId().value);
        Bytes.putInt((byte[])other, (int)8, (int)count);
        return new stateid4(other, 0);
    }

    public sessionid4 createSessionId(NFS4Client client, int sequence) {
        byte[] id = new byte[16];
        Bytes.putLong((byte[])id, (int)0, (long)client.getId().value);
        Bytes.putInt((byte[])id, (int)12, (int)sequence);
        return new sessionid4(id);
    }

    public int getLeaseTime() {
        return this._leaseTime;
    }

    private static final class DeadClientCollector
    extends NopCacheEventListener<clientid4, NFS4Client> {
        private final ClientRecoveryStore clientStore;

        private DeadClientCollector(ClientRecoveryStore clientStore) {
            this.clientStore = clientStore;
        }

        @Override
        public void notifyExpired(Cache<clientid4, NFS4Client> cache, NFS4Client client) {
            _log.info("Removing expired client: {}", (Object)client);
            client.tryDispose();
            this.clientStore.removeClient(client.getOwnerId());
        }
    }
}

