/*
 * Decompiled with CFR 0.152.
 */
package org.dcache.nfs;

import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Ordering;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.net.InetAddress;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.dcache.nfs.ExportTable;
import org.dcache.nfs.FsExport;
import org.dcache.nfs.HostEntryComparator;
import org.dcache.nfs.v4.xdr.layouttype4;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExportFile
implements ExportTable {
    private static final Logger _log = LoggerFactory.getLogger(ExportFile.class);
    private volatile ImmutableMultimap<Integer, FsExport> _exports;
    private final Callable<URI[]> _exportFileProvider;

    public ExportFile(File file) throws IOException {
        this(file, null);
    }

    public ExportFile(File file, File dir) throws IOException {
        this._exportFileProvider = () -> {
            if (dir != null && dir.exists()) {
                Preconditions.checkArgument((boolean)dir.isDirectory(), (Object)(dir.getAbsolutePath() + " exist but not a directory"));
                File[] files = dir.listFiles((d, n) -> n.endsWith(".exports") && n.charAt(0) != '.');
                URI[] exportFiles = new URI[files.length + 1];
                exportFiles[0] = file.toURI();
                for (int i = 0; i < files.length; ++i) {
                    exportFiles[i + 1] = files[i].toURI();
                }
                return exportFiles;
            }
            return new URI[]{file.toURI()};
        };
        this.rescan();
    }

    public ExportFile(URI uri) throws IOException {
        this._exportFileProvider = () -> new URI[]{uri};
        this.rescan();
    }

    public ExportFile(Reader reader) throws IOException {
        this._exportFileProvider = () -> {
            throw new IllegalStateException("exports uri not set, rescan impossible");
        };
        this._exports = ExportFile.parse(reader);
    }

    @Override
    public Stream<FsExport> exports() {
        return this._exports.values().stream();
    }

    private static ImmutableMultimap<Integer, FsExport> parse(Reader reader) throws IOException {
        List<String> lines;
        try (BufferedReader bufferedReader = new BufferedReader(reader);){
            lines = bufferedReader.lines().collect(Collectors.toList());
        }
        return ExportFile.parseExportLines(lines);
    }

    private static ImmutableMultimap<Integer, FsExport> parse(URI ... exportFiles) throws IOException {
        ImmutableListMultimap.Builder exportsBuilder = ImmutableListMultimap.builder();
        for (URI exportFile : exportFiles) {
            List<String> lines = Files.readAllLines(Paths.get(exportFile));
            ImmutableMultimap<Integer, FsExport> export = ExportFile.parseExportLines(lines);
            exportsBuilder.putAll(export);
        }
        return exportsBuilder.orderValuesBy((Comparator)Ordering.from(HostEntryComparator::compare).onResultOf(FsExport::client)).build();
    }

    private static ImmutableMultimap<Integer, FsExport> parseExportLines(Iterable<String> lines) throws IOException {
        ImmutableListMultimap.Builder exportsBuilder = ImmutableListMultimap.builder();
        for (String line : lines) {
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') continue;
            if (line.charAt(0) != '/') {
                _log.warn("Ignoring entry with non absolute export path: " + line);
                continue;
            }
            int pathEnd = line.indexOf(32);
            if (pathEnd < 0) {
                FsExport export = new FsExport.FsExportBuilder().build(line);
                exportsBuilder.put((Object)export.getIndex(), (Object)export);
                continue;
            }
            String path = line.substring(0, pathEnd);
            Splitter splitter = Splitter.on((char)' ').omitEmptyStrings().trimResults();
            for (String hostAndOptions : splitter.split((CharSequence)line.substring(pathEnd + 1))) {
                try {
                    FsExport.FsExportBuilder exportBuilder = new FsExport.FsExportBuilder();
                    Iterator s = Splitter.on((CharMatcher)CharMatcher.anyOf((CharSequence)"(,)")).omitEmptyStrings().trimResults().split((CharSequence)hostAndOptions).iterator();
                    String host = (String)s.next();
                    exportBuilder.forClient(host);
                    while (s.hasNext()) {
                        String option = (String)s.next();
                        if (option.equals("rw")) {
                            exportBuilder.rw();
                            continue;
                        }
                        if (option.equals("ro")) {
                            exportBuilder.ro();
                            continue;
                        }
                        if (option.equals("root_squash")) {
                            exportBuilder.notTrusted();
                            continue;
                        }
                        if (option.equals("no_root_squash")) {
                            exportBuilder.trusted();
                            continue;
                        }
                        if (option.equals("acl")) {
                            exportBuilder.withAcl();
                            continue;
                        }
                        if (option.equals("noacl") || option.equals("no_acl")) {
                            exportBuilder.withoutAcl();
                            continue;
                        }
                        if (option.equals("all_squash")) {
                            exportBuilder.allSquash();
                            continue;
                        }
                        if (option.startsWith("sec=")) {
                            String secFlavor = option.substring(4);
                            exportBuilder.withSec(FsExport.Sec.valueOf(secFlavor.toUpperCase()));
                            continue;
                        }
                        if (option.startsWith("anonuid=")) {
                            int anonuid = Integer.parseInt(option.substring(8));
                            exportBuilder.withAnonUid(anonuid);
                            continue;
                        }
                        if (option.startsWith("anongid=")) {
                            int anongid = Integer.parseInt(option.substring(8));
                            exportBuilder.withAnonGid(anongid);
                            continue;
                        }
                        if (option.equals("dcap")) {
                            exportBuilder.withDcap();
                            continue;
                        }
                        if (option.equals("no_dcap")) {
                            exportBuilder.withoutDcap();
                            continue;
                        }
                        if (option.equals("all_root")) {
                            exportBuilder.withAllRoot();
                            continue;
                        }
                        if (option.equals("pnfs")) {
                            exportBuilder.withPnfs();
                            continue;
                        }
                        if (option.equals("nopnfs") || option.equals("no_pnfs")) {
                            exportBuilder.withoutPnfs();
                            continue;
                        }
                        if (option.startsWith("lt=")) {
                            Iterable lt = Splitter.on((String)":").omitEmptyStrings().split((CharSequence)option.substring(3));
                            StreamSupport.stream(lt.spliterator(), false).map(String::toUpperCase).map(t -> "LAYOUT4_" + t).map(layouttype4::valueOf).forEach(exportBuilder::withLayoutType);
                            continue;
                        }
                        if (option.equals("secure")) {
                            exportBuilder.withPrivilegedClientPort();
                            continue;
                        }
                        if (option.equals("insecure")) {
                            exportBuilder.withoutPrivilegedClientPort();
                            continue;
                        }
                        throw new IllegalArgumentException("Unsupported option: " + option);
                    }
                    FsExport export = exportBuilder.build(path);
                    exportsBuilder.put((Object)export.getIndex(), (Object)export);
                }
                catch (IllegalArgumentException e) {
                    _log.error("Invalid export entry [" + hostAndOptions + "] : " + e.getMessage());
                }
            }
        }
        return exportsBuilder.orderValuesBy((Comparator)Ordering.from(HostEntryComparator::compare).onResultOf(FsExport::client)).build();
    }

    @Override
    public FsExport getExport(String path, InetAddress client) {
        String normalizedPath = FsExport.normalize(path);
        return this.getExport(FsExport.getExportIndex(normalizedPath), client);
    }

    @Override
    public FsExport getExport(int index, InetAddress client) {
        for (FsExport export : this._exports.get((Object)index)) {
            if (!export.isAllowed(client)) continue;
            return export;
        }
        return null;
    }

    @Override
    public Stream<FsExport> exports(InetAddress client) {
        return this._exports.values().stream().filter(e -> e.isAllowed(client)).sorted((Comparator<FsExport>)Ordering.from(HostEntryComparator::compare).onResultOf(FsExport::client));
    }

    public final void rescan() throws IOException {
        try {
            this._exports = ExportFile.parse(this._exportFileProvider.call());
        }
        catch (Exception e) {
            Throwables.throwIfInstanceOf((Throwable)e, IOException.class);
            Throwables.throwIfUnchecked((Throwable)e);
            throw new RuntimeException("Unhandled exception", e);
        }
    }
}

